<?php
/**
 * HRM Attendance Debug Script
 * 
 * Usage: Access this file directly in browser while logged in to WordPress
 * URL: /wp-content/plugins/hr-management/debug_attendance.php
 * 
 * This will show the complete state of attendance for the current user.
 */

// Load WordPress
$wp_load_path = dirname(dirname(dirname(dirname(__FILE__)))) . '/wp-load.php';
if (!file_exists($wp_load_path)) {
    die('WordPress not found. Please place this plugin in wp-content/plugins/');
}
require_once($wp_load_path);

// Security check - only admins can run this
if (!current_user_can('manage_options')) {
    die('Access denied. Admins only.');
}

global $wpdb;
$current_user = wp_get_current_user();

echo "<h1>HRM Attendance Debug Report</h1>";
echo "<p>Generated: " . current_time('mysql') . "</p>";

// 1. Check hrm_ajax localized script
echo "<h2>1. Localized Script Check</h2>";
echo "<pre>";
echo "hrm_ajax object should contain:\n";
echo "- ajaxurl: " . admin_url('admin-ajax.php') . "\n";
echo "- nonce: " . wp_create_nonce('hrm_ajax_nonce') . "\n";
echo "</pre>";

// 2. Current User Info
echo "<h2>2. Current User Info</h2>";
echo "<pre>";
echo "User ID: " . $current_user->ID . "\n";
echo "Username: " . $current_user->user_login . "\n";
echo "Email: " . $current_user->user_email . "\n";
echo "Display Name: " . $current_user->display_name . "\n";
echo "</pre>";

// 3. Employee Lookup
echo "<h2>3. Employee Lookup</h2>";
$post_type = class_exists('HRM_Setup') ? HRM_Setup::EMPLOYEE_POST_TYPE : 'hrm_employee';

// By Author
$employee_by_author = $wpdb->get_var($wpdb->prepare(
    "SELECT ID FROM {$wpdb->posts} WHERE post_author = %d AND post_type = %s AND post_status = 'publish' LIMIT 1",
    $current_user->ID, $post_type
));

// By Email
$employee_by_email = $wpdb->get_var($wpdb->prepare(
    "SELECT p.ID FROM {$wpdb->posts} p 
     INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
     WHERE p.post_type = %s AND p.post_status = 'publish' 
     AND pm.meta_key = 'hrm_email' AND pm.meta_value = %s LIMIT 1",
    $post_type, $current_user->user_email
));

echo "<pre>";
echo "Post Type: $post_type\n";
echo "Employee by Author (post_author = {$current_user->ID}): " . ($employee_by_author ?: 'NOT FOUND') . "\n";
echo "Employee by Email ({$current_user->user_email}): " . ($employee_by_email ?: 'NOT FOUND') . "\n";
echo "</pre>";

$employee_id = $employee_by_author ?: $employee_by_email;

if ($employee_id) {
    echo "<p style='color: green;'>✓ Employee found: ID $employee_id</p>";
    
    // 4. Today's Attendance
    echo "<h2>4. Today's Attendance Record</h2>";
    $today = current_time('Y-m-d');
    $attendance = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s",
        $employee_id, $today
    ));
    
    echo "<pre>";
    echo "Date: $today\n";
    echo "Records found: " . count($attendance) . "\n\n";
    
    if ($attendance) {
        foreach ($attendance as $i => $record) {
            echo "Record #" . ($i + 1) . ":\n";
            echo "  - ID: {$record->id}\n";
            echo "  - Check In: " . ($record->check_in ?: 'NULL') . "\n";
            echo "  - Check Out: " . ($record->check_out ?: 'NULL') . "\n";
            echo "  - Status: {$record->status}\n";
            echo "  - Hours: {$record->hours_worked}\n";
            echo "\n";
        }
    } else {
        echo "No attendance records for today.\n";
        echo "First check-in will create a NEW record.\n";
    }
    echo "</pre>";
    
    // 5. All Attendance Records (last 10)
    echo "<h2>5. Recent Attendance History (Last 10)</h2>";
    $history = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d ORDER BY date DESC, id DESC LIMIT 10",
        $employee_id
    ));
    
    if ($history) {
        echo "<table border='1' cellpadding='5'>";
        echo "<tr><th>ID</th><th>Date</th><th>Check In</th><th>Check Out</th><th>Status</th><th>Hours</th></tr>";
        foreach ($history as $row) {
            echo "<tr>";
            echo "<td>{$row->id}</td>";
            echo "<td>{$row->date}</td>";
            echo "<td>" . ($row->check_in ?: '-') . "</td>";
            echo "<td>" . ($row->check_out ?: '-') . "</td>";
            echo "<td>{$row->status}</td>";
            echo "<td>{$row->hours_worked}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p>No attendance history found.</p>";
    }
    
} else {
    echo "<p style='color: red;'>✗ NO EMPLOYEE RECORD FOUND!</p>";
    echo "<p>This user cannot check in because there's no matching employee record.</p>";
    
    echo "<h3>Troubleshooting:</h3>";
    echo "<ol>";
    echo "<li>Go to HR Management → Employees</li>";
    echo "<li>Create or edit an employee</li>";
    echo "<li>Make sure the employee's WordPress user is set to this user (ID: {$current_user->ID})</li>";
    echo "<li>OR make sure the employee's email matches: {$current_user->user_email}</li>";
    echo "</ol>";
    
    // Show all employees for reference
    echo "<h3>All Employees in System:</h3>";
    $all_employees = $wpdb->get_results($wpdb->prepare(
        "SELECT p.ID, p.post_title, p.post_author, pm.meta_value as email 
         FROM {$wpdb->posts} p 
         LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = 'hrm_email'
         WHERE p.post_type = %s AND p.post_status = 'publish'",
        $post_type
    ));
    
    if ($all_employees) {
        echo "<table border='1' cellpadding='5'>";
        echo "<tr><th>Employee ID</th><th>Name</th><th>WP User ID (Author)</th><th>Email</th></tr>";
        foreach ($all_employees as $emp) {
            echo "<tr>";
            echo "<td>{$emp->ID}</td>";
            echo "<td>{$emp->post_title}</td>";
            echo "<td>{$emp->post_author}</td>";
            echo "<td>" . ($emp->email ?: '-') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p>No employees found in the system!</p>";
    }
}

// 6. Debug Log
echo "<h2>6. Debug Log (Last 50 lines)</h2>";
$debug_log_path = HRM_PLUGIN_PATH . 'hrm_debug.log';
if (file_exists($debug_log_path)) {
    $log_content = file_get_contents($debug_log_path);
    $lines = explode("\n", $log_content);
    $last_lines = array_slice($lines, -50);
    echo "<pre style='background: #f5f5f5; padding: 10px; max-height: 400px; overflow-y: auto;'>";
    echo htmlspecialchars(implode("\n", $last_lines));
    echo "</pre>";
} else {
    echo "<p>No debug log found. It will be created on first AJAX request.</p>";
}

// 7. Quick Test Button
echo "<h2>7. Quick Test</h2>";
echo "<p>Open browser console (F12) and paste this to test AJAX:</p>";
$test_nonce = wp_create_nonce('hrm_ajax_nonce');
echo "<pre style='background: #f5f5f5; padding: 10px;'>";
echo htmlspecialchars("
// Test Check-In AJAX
const formData = new FormData();
formData.append('action', 'hrm_mark_attendance');
formData.append('type', 'check_in');
formData.append('nonce', '$test_nonce');

fetch('" . admin_url('admin-ajax.php') . "', {
    method: 'POST',
    body: formData
})
.then(r => r.text())
.then(text => {
    console.log('Response:', text);
    try {
        const data = JSON.parse(text);
        console.log('Parsed:', data);
    } catch(e) {
        console.error('Parse error:', e);
    }
})
.catch(e => console.error('Fetch error:', e));
");
echo "</pre>";

echo "<hr><p><em>Debug script completed.</em></p>";
